// src/pages/TrainingDetailPage.jsx
import { useEffect, useState } from "react";
import { useNavigate, useParams } from "react-router-dom";
import api from "../api/api";
import { useAuth } from "../AuthContext";
import logoCikos from "../assets/logo-cikos.png"; // sesuaikan nama file

export default function TrainingDetailPage() {
  const { id } = useParams();
  const navigate = useNavigate();
  const { user } = useAuth();
  const [data, setData] = useState(null);
  const [loading, setLoading] = useState(true);

  const loadData = async () => {
    setLoading(true);
    const res = await api.get(`/training-requests/${id}`);
    setData(res.data);
    setLoading(false);
  };

  useEffect(() => {
    loadData();
  }, [id]);

  const handlePrint = () => {
    window.print();
  };

  if (loading) return <div style={{ padding: 20 }}>Loading...</div>;
  if (!data) return <div style={{ padding: 20 }}>Data tidak ditemukan.</div>;

  const purposes = data.purposes || [];
  const createdAt = data.created_at
    ? new Date(data.created_at).toLocaleDateString("id-ID")
    : "-";
  const trainingDate = data.training_date
    ? new Date(data.training_date).toLocaleDateString("id-ID")
    : "-";

  const pemohon = data.user || {};

  // helper tampilan status
  const statusLabel = (() => {
    switch (data.status) {
      case "submitted":
        return "Diajukan (menunggu persetujuan HRD)";
      case "approved_hrd":
        return "Disetujui HRD (menunggu Direktur bila biaya > 20.000.000)";
      case "rejected_hrd":
        return "Ditolak HRD";
      case "approved_director":
        return "Disetujui HRD & Direktur";
      case "rejected_director":
        return "Disetujui HRD, ditolak Direktur";
      default:
        return data.status;
    }
  })();

  const formatRupiah = (v) =>
    typeof v === "number"
      ? v.toLocaleString("id-ID", { style: "currency", currency: "IDR" })
      : "-";

  return (
    <div className="print-page">
      {/* Tombol yang hanya muncul di layar, disembunyikan saat print */}
      <div className="no-print" style={{ margin: "10px 0" }}>
        <button onClick={() => navigate(-1)} style={{ marginRight: 10 }}>
          &larr; Kembali
        </button>
        <button onClick={handlePrint}>Print</button>
        <span style={{ marginLeft: 20 }}>
          Login sebagai: <strong>{user?.name}</strong> ({user?.role})
        </span>
      </div>

      {/* Header dokumen */}
      <header className="doc-header">
        <div className="doc-header-left">
          <img src={logoCikos} alt="Logo PT Cikos Indonesia" />
        </div>
        <div className="doc-header-center">
          <h2>PT. CIKOS INDONESIA</h2>
          <h3>FORM PERMOHONAN / PENGAJUAN TRAINING</h3>
        </div>
        <div className="doc-header-right">
          <table>
            <tbody>
              <tr>
                <td>No. Dokumen</td>
                <td>:</td>
                <td>FR-HRGA-032</td>
              </tr>
              <tr>
                <td>Tgl Dibuat</td>
                <td>:</td>
                <td>13/06/2024</td>
              </tr>
              <tr>
                <td>Revisi</td>
                <td>:</td>
                <td>0</td>
              </tr>
              <tr>
                <td>Hal</td>
                <td>:</td>
                <td>1 dari 1</td>
              </tr>
            </tbody>
          </table>
        </div>
      </header>

      {/* Info status */}
      <div className="doc-status">
        <strong>Status:</strong> {statusLabel}
      </div>

      {/* Identitas pemohon */}
      <section className="doc-section">
        <h4>DATA PEMOHON</h4>
        <table className="doc-table">
          <tbody>
            <tr>
              <td style={{ width: "20%" }}>Nama</td>
              <td style={{ width: "3%" }}>:</td>
              <td>{pemohon.name || "-"}</td>
            </tr>
            <tr>
              <td>Tanggal Pengajuan</td>
              <td>:</td>
              <td>{createdAt}</td>
            </tr>
            <tr>
              <td>Jabatan</td>
              <td>:</td>
              <td>{pemohon.position || "-"}</td>
            </tr>
            <tr>
              <td>Departemen</td>
              <td>:</td>
              <td>{pemohon.department || "-"}</td>
            </tr>
          </tbody>
        </table>
      </section>

      {/* Data Training */}
      <section className="doc-section">
        <h4>DATA TRAINING</h4>
        <table className="doc-table">
          <tbody>
            <tr>
              <td style={{ width: "20%" }}>Nama Training</td>
              <td style={{ width: "3%" }}>:</td>
              <td>{data.title}</td>
            </tr>
            <tr>
              <td>Waktu</td>
              <td>:</td>
              <td>{trainingDate}</td>
            </tr>
            <tr>
              <td>Penyelenggara</td>
              <td>:</td>
              <td>{data.vendor || "-"}</td>
            </tr>
            <tr>
              <td>Biaya</td>
              <td>:</td>
              <td>{formatRupiah(data.cost)}</td>
            </tr>
          </tbody>
        </table>
      </section>

      {/* Tujuan Training */}
      <section className="doc-section">
        <h4>TUJUAN TRAINING</h4>
        <div className="doc-checkbox-group">
          <CheckboxLine
            label="Meningkatkan keterampilan"
            checked={purposes.includes("skill")}
          />
          <CheckboxLine
            label="Meningkatkan pengetahuan manajerial"
            checked={purposes.includes("managerial")}
          />
          <CheckboxLine
            label="Memenuhi kriteria Job Description"
            checked={purposes.includes("jobdesc")}
          />
          <CheckboxLine
            label="Penyegaran / Refreshment"
            checked={purposes.includes("refreshment")}
          />
          {/* Kalau nanti kamu mau 'lain-lain', tinggal tambah item lagi */}
        </div>
      </section>

      {/* Hasil yang diharapkan */}
      <section className="doc-section">
        <h4>
          Hasil yang bisa Peserta perlihatkan/sumbangkan pada Perusahaan
          (dalam bentuk nyata)
        </h4>
        <div className="doc-textbox">
          {data.expected_result || "-"}
        </div>
      </section>

      {/* Saran HRD */}
      <section className="doc-section">
        <h4>SARAN HRD</h4>
        <div className="doc-textbox">
          {data.hrd_notes || "-"}
        </div>
      </section>

      {/* Tanda tangan */}
      <section className="doc-section doc-signatures">
        <div className="doc-signature-column">
          <div>Peserta</div>
          <div className="doc-signature-box">
            {/* ruang tanda tangan */}
          </div>
          <div>
            <strong>{pemohon.name || "(Nama Peserta)"}</strong>
          </div>
        </div>

        <div className="doc-signature-column">
          <div>Disetujui oleh (HRGA)</div>
          <div className="doc-signature-box" />
          <div>
            <strong>
              {data.approved_by_hrd && data.status !== "rejected_hrd"
                ? "HRGA"
                : "(Belum disetujui)"}
            </strong>
          </div>
        </div>

        <div className="doc-signature-column">
          <div>Diketahui oleh (Direktur)</div>
          <div className="doc-signature-box" />
          <div>
            <strong>
              {data.approved_by_director &&
              data.status === "approved_director"
                ? "Direktur"
                : "(Belum diketahui)"}
            </strong>
          </div>
        </div>
      </section>
    </div>
  );
}

function CheckboxLine({ label, checked }) {
  return (
    <div className="doc-checkbox-line">
      <div className="doc-checkbox">{checked ? "☑" : "☐"}</div>
      <span>{label}</span>
    </div>
  );
}